/*
        This code is released under the zlib/libpng license.

        Copyright (C) 2007 Christian Kamm (kamm incasoftware de)

        This software is provided 'as-is', without any express or implied
        warranty.  In no event will the authors be held liable for any damages
        arising from the use of this software.

        Permission is granted to anyone to use this software for any purpose,
        including commercial applications, and to alter it and redistribute it
        freely, subject to the following restrictions:

        1. The origin of this software must not be misrepresented; you must not
           claim that you wrote the original software. If you use this software
           in a product, an acknowledgment in the product documentation would be
           appreciated but is not required.
        2. Altered source versions must be plainly marked as such, and must not be
           misrepresented as being the original software.
        3. This notice may not be removed or altered from any source distribution.
*/

#ifndef INCLUDE_MATLABACCESSORS
#define INCLUDE_MATLABACCESSORS

#include <mat.h>
#include <string>
#include <vector>

namespace MatlabUtilities {
  
  /// typedefs for matlab types
  namespace Types
  {
    // typedefs for some matlab types
    struct real {};
    struct integer {};
    struct string {};
    struct int_vector {};
    struct real_vector {};
    template <int N> struct fixed_length_real_vector {};
  }
  
  /// small accessor functions that offer reading, writing and typechecking functionality
  namespace Accessors 
  {
    
    /// specializations of this template provide typechecking
    template <typename matlab_type>
    struct TypeCheck
    {
      bool typeCheck(mxArray* var);
    };
    
    /// specializations of this template know how to read a particular
    /// matlab type into result type
    template <typename matlab_type, typename result_type>
    struct Reader
    {
      void read(mxArray* var, result_type& result); 
    };
    
    /// specializations of this template know how to write a particular
    /// source type into a matlab type
    template <typename matlab_type, typename source_type>
    struct Writer
    {
      mxArray* write(source_type what);
    };
    
    
    //
    // implementation for noncomplex scalars
    //
    
    template <>
    struct Reader<Types::real, double>
    {
      static void read(mxArray* var, double& result)
      {
        result = mxGetScalar(var);
      }
    };
    
    template <>
    struct Writer<Types::real, double>
    {
      static mxArray* write(double what)
      {
        mxArray* var = mxCreateDoubleMatrix(1, 1, mxREAL);
        
        if(!var)
          return NULL;
        
        double* raw = mxGetPr(var);
        *raw = what;
        
        return var;
      }
    };
    
    template <>
    struct TypeCheck<Types::real>
    {
      static bool typeCheck(mxArray* var)
      {
        if(mxIsDouble(var) && !mxIsComplex(var) &&
              mxGetN(var)*mxGetM(var) == 1)
          return true;
        else
          return false;
      }
    };
    
    
    //
    // implementation for integers
    //
    
    template <>
    struct Reader<Types::integer, int>
    {
      static void read(mxArray* var, int& result)
      {
        double temp;
        Reader<Types::real, double>::read(var, temp);
        result = static_cast<int>(temp);
      }
    };
    
    template <>
    struct Writer<Types::integer, int>
    {
      static mxArray* write(int what)
      {
        return Writer<Types::real, double>::write(what);
      }
    };

    template <>
    struct TypeCheck<Types::integer>
    {
      static bool typeCheck(mxArray* var)
      {
        return TypeCheck<Types::real>::typeCheck(var);
      }
    };
    
    
    //
    // implementation for strings
    //
    
    template <>
    struct Reader<Types::string, std::string>
    {
      static void read(mxArray* var, std::string& result)
      {
        std::string::size_type size = mxGetN(var) + 1;
        char* buffer = new char[size];
        mxGetString(var, buffer, size);
        result = buffer;
        delete [] buffer;
      }
    };
    
    template <>
    struct Writer<Types::string, const char*>
    {
      static mxArray* write(const char* what)
      {
        return mxCreateString(what);
      }
    };
    
    template <int N>
    struct Writer<Types::string, const char [N]>
    {
      static mxArray* write(const char what[N])
      {
        return mxCreateString(what);
      }
    };
    
    template <>
    struct Writer<Types::string, std::string>
    {
      static mxArray* write(std::string& what)
      {
        return mxCreateString(what.c_str());
      }
    };
    
    template <>
    struct TypeCheck<Types::string>
    {
      static bool typeCheck(mxArray* var)
      {
        if(mxIsChar(var))
          return true;
        else
          return false;
      }
    };
    
    
    //
    // implementation for real vectors
    //
    
    template <>
    struct Reader<Types::real_vector, std::vector<double> >
    {
      static void read(mxArray* var, std::vector<double>& result)
      {
        result.clear();
        size_t n_elements = mxGetM(var) * mxGetN(var);
        result.reserve(n_elements);
        
        double *ptr = mxGetPr(var);
        for(int i = 0; i < n_elements; ++i)
          result.push_back(ptr[i]);
      }
    };
    
    template <>
    struct Writer<Types::real_vector, std::vector<double> >
    {
      static mxArray* write(std::vector< double >& what)
      {
        int dims[1];
        dims[0] = what.size();
        
        mxArray* var = mxCreateNumericArray(1, dims, mxDOUBLE_CLASS, mxREAL);
        
        if(!var)
          return NULL;
        
        double* raw = mxGetPr(var);
        for(int i = 0; i < dims[0]; ++i)
          raw[i] = what[i];
        
        return var;
      }
    };
    
    template <>
    struct TypeCheck<Types::real_vector>
    {
      static bool typeCheck(mxArray* var)
      {
        if(mxIsDouble(var) && !mxIsComplex(var) &&
              (mxGetM(var) == 1 || mxGetN(var) == 1))
          return true;
        else
          return false;
      }
    };
    
    
    //
    // implementation for int vectors
    //
    
    template <>
    struct Reader<Types::int_vector, std::vector<int> >
    {
      static void read(mxArray* var, std::vector<int>& result)
      {
        result.clear();
        size_t n_elements = mxGetM(var) * mxGetN(var);
        result.reserve(n_elements);
        
        double *ptr = mxGetPr(var);
        for(int i = 0; i < n_elements; ++i)
          result.push_back(static_cast<int>(ptr[i]));
      }
    };
    
    template <>
    struct TypeCheck<Types::int_vector>
    {
      static bool typeCheck(mxArray* var)
      {
        if(mxIsDouble(var) && !mxIsComplex(var) &&
              (mxGetM(var) == 1 || mxGetN(var) == 1))
          return true;
        else
          return false;
      }
    };
    
    
    //
    // implementation for fixed length real vectors
    //
    
    template <int N>
    struct Reader< Types::fixed_length_real_vector<N>, std::vector<double> >
    {
      static void read(mxArray* var, std::vector<double>& result)
      {
        result.clear();
        result.reserve(N);
        double *ptr = mxGetPr(var);
        for(int i = 0; i < N; ++i)
          result.push_back(ptr[i]);
      }
    };
    
    template <int N>
    struct Reader< Types::fixed_length_real_vector<N>, double[N] >
    {
      static void read(mxArray* var, double result[N])
      {
        double *ptr = mxGetPr(var);
        for(int i = 0; i < N; ++i)
          result[i] = ptr[i];
      }
    };
    
    template <int N>
    struct TypeCheck< Types::fixed_length_real_vector<N> >
    {
      static bool typeCheck(mxArray* var)
      {
        if( mxIsDouble(var) && 
            !mxIsComplex(var) &&
            (mxGetM(var) == 1 || mxGetN(var) == 1) &&
            mxGetM(var) * mxGetN(var) == N
              )
          return true;
        else
          return false;
      }
    };
  }
}

#endif
