/*
        This code is released under the zlib/libpng license.

        Copyright (C) 2007 Christian Kamm (kamm incasoftware de)

        This software is provided 'as-is', without any express or implied
        warranty.  In no event will the authors be held liable for any damages
        arising from the use of this software.

        Permission is granted to anyone to use this software for any purpose,
        including commercial applications, and to alter it and redistribute it
        freely, subject to the following restrictions:

        1. The origin of this software must not be misrepresented; you must not
           claim that you wrote the original software. If you use this software
           in a product, an acknowledgment in the product documentation would be
           appreciated but is not required.
        2. Altered source versions must be plainly marked as such, and must not be
           misrepresented as being the original software.
        3. This notice may not be removed or altered from any source distribution.
*/

#ifndef INCLUDE_MATLABFILE
#define INCLUDE_MATLABFILE

#include <mat.h>

#include <exception>
#include <stdexcept>
#include <string>
#include <vector>

#include "matlabaccessors.h"

/// utilities for dealing with matlab and its c api
namespace MatlabUtilities
{
  /// mat-file access class
  /**
   * This class allows easy access to mat-files.
   */
  class File
  {
    public:
      /// construct using a filename and whether we should overwrite an existing file (and thus open writeonly)
      File(const char* filename, bool overwrite = false);
      
      /// non-virtual destructor: don't derive from this class
      ~File();
      
      /// flag for deciding what to do if variable is not found in file
      struct not_found_action { enum action { throw_exception, ignore }; };
      
      /// read from file, throw if not compatible or if it doesn't exist and not_found_action is throw_exception
      template <typename matlab_type, typename return_type>
      void read(const char* name, return_type& result, not_found_action::action on_not_found = not_found_action::throw_exception);
      
      /// write to file
      template <typename matlab_type, typename input_type>
      void write(const char* name, input_type& what);
      
      /// determine whether a variable exists in the file
      bool variableExists(const char* name) const;
      
      //
      // exceptions
      //
      
    private:
      /// runtime error exception that takes a string parameter and adds another before it
      class StringException : public std::runtime_error
      {
        public:
          StringException(const char* description, const char* parameter)
            : std::runtime_error(std::string(description) + std::string(parameter)) {}
          virtual ~StringException() throw() {}
          
//          virtual const char* what() const throw() { return (description_ + parameter_).c_str(); }
      };
      
    public:    
      /// thrown when opening the file failed
      class OpenFailedException : public StringException
      {
        public:
          OpenFailedException(const char* filename) 
            : StringException("mat-File open failed: ", filename) {}
      };
      
      /// thrown when trying to read a variable that doesn't exist
      class VariableDoesNotExistException : public StringException
      {
        public:
          VariableDoesNotExistException(const char* varname) 
            : StringException("Variable does not exist: ", varname) {}
      };
      
      /// thrown when the variable being read isn't of the type requested
      class VariableHasWrongTypeException : public StringException
      {
        public:
          VariableHasWrongTypeException(const char* varname) 
          : StringException("Variable has wrong type: ", varname) {}
      };
    
      /// thrown when a write failed
      class WriteFailedException : public StringException
      {
        public:
          WriteFailedException(const char* varname) 
          : StringException("Write to mat-file failed: ", varname) {}
      };
    
      /// thrown when a variable for writing couldn't be created
      class VariableCreationFailedException : public StringException
      {
        public:
          VariableCreationFailedException(const char* varname) 
          : StringException("Could not create variable for writing: ", varname) {}
      };
      
      /// thrown when trying to read from a write-only file
      class WriteOnlyException : public std::logic_error 
      {
        public:
          WriteOnlyException() throw() : std::logic_error("Tried to read from write-only file.") {}
      };
    
    private:
      /// gets a variable by name
      mxArray* getVariable(const char* name) const;
      
      /// matlab file handle
      MATFile* mat_file_;
      
      /// was the file opened writeonly?
      bool write_only_;
  };
}


//
// file implementation
//

namespace MatlabUtilities
{
  File::File(const char* filename, bool overwrite)
    : write_only_(false)
  {
    if(!overwrite)
      mat_file_ = matOpen(filename, "u");
    else
      mat_file_ = matOpen(filename, "w");
    
    if(!mat_file_)
      throw OpenFailedException(filename);
  }
  
  File::~File()
  {
    if(mat_file_)
      matClose(mat_file_); 
  }
  
  template <typename matlab_type, typename return_type>
  void File::read(const char* name, return_type& result, not_found_action::action on_not_found)
  {
    if(write_only_)
      throw WriteOnlyException();
    
    mxArray* var = getVariable(name);
    
    if(!var)
      if(on_not_found == not_found_action::ignore)
        return;
      else
        throw VariableDoesNotExistException(name);
    
    if(!Accessors::TypeCheck<matlab_type>::typeCheck(var)) {
      mxDestroyArray(var);
      throw VariableHasWrongTypeException(name);
    }
    
    Accessors::Reader<matlab_type, return_type>::read(var, result);
    
    mxDestroyArray(var);
  }
  
  template <typename matlab_type, typename input_type>
  void File::write(const char* name, input_type& what)
  {
    mxArray* var = Accessors::Writer<matlab_type, input_type>::write(what);
    
    if(!var)
      throw VariableCreationFailedException(name);
    
    int status = matPutVariable(mat_file_, name, var);
    if(status != 0) {
      mxDestroyArray(var);
      throw WriteFailedException(name);
    }
  
    mxDestroyArray(var);
  }
  
  bool File::variableExists(const char* name) const
  {
    if(write_only_)
      throw WriteOnlyException();
    
    mxArray* var = getVariable(name);
    
    if(!var)
      return false;
    
    mxDestroyArray(var);
    return true;
  }
  
  mxArray* File::getVariable(const char* name) const
  {
    return matGetVariable(mat_file_, name);
  }

}


#endif
